/*  Ephmeris.c */
/*  Converted to TurboC from BASIC program EPHM2.BAS, originally written by
    David Bulger.
    Converted by:  David A. Beisel [70277,3600] */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <dos.h>
#include "ephmeris.h"

/****************** Global Variables ********************/

FILE *outfile;
char *filename;
const double PI  = 3.141592653589793;
const double P2  = 6.283185308;
const double EPOCH = 2415020;
double t, t2, t3, p, ra, dc, ra1, ra2, ra3, dc1, dc2, dc3, r;
double mag, mag1, mag2, mag3, mag4, pull;
double xsol, ysol, zsol, solrad, oblq;
double eccent, mean, ea1, ma, ma1, ma2, v0, julian;
char *o;
float dy, month, day, year, hour, minute, second;
int prn;
double a, e, i, u, w;
/********************************************************/

double fndegrad (double x)
  {
    return (x * (PI / 180));
  }

double fnraddeg (double x)
  {
    return (x * (180 / PI));
  }

double fnarccos (double x)
  {
    return (-atan(x/sqrt(-x * x + 1)) + PI/2);
  }

double fnmodulo (double x, double y)
  {
    return (y * (x / y - (long int) (x/y)));
  }

double fns (double x)
  {
    return (sin(180/ PI * x));
  }

double fnu (double x)
  {
    return (x - ((long int) (x/360) * 360));
  }

int sgn (double x)
  {
    if (x >= 0)
      return (1);
    else
      return (-1);
  }

double fnp (double x)
  {
    return (sgn(x) * ((fabs(x) / 3600) / 360 - (long int) ((fabs(x)/3600)/360)) * 360);
  }

double fnacos (double x)
  {
    return (-atan(x/sqrt(-x*x+1)) + PI / 2);
  }

double fnasin (double x)
  {
    return (atan(x/sqrt(-x*x+1)));
  }



/*** Solar Elements Update ***/

void solael ()

  {
    double asol, esol, isol, usol, wsol, msol;
    double vsol, rsol, rasol, dcsol;

    asol = 1.00000003;
    esol = 1.675104E-2 - .0000418*t - 1.26E-7 * t2;
    wsol = 4.908229467 + .0300052642 * t + 7.902463E-6 * t2 +
           .0000000058117706 * t3;
    oblq = .4093197552 - 2.27111E-04 * t - .00000028604007 * t2 +
           8.775128E-6 * t3;
    msol = 628.3019457 * t;
    msol = fnmodulo (msol,P2);
    msol = msol + 6.256583774;
    msol = fnmodulo (msol,P2);
    msol = msol -2.618E-6 * t2 - .0000000581776417 * t3;
    msol = fnmodulo (msol, P2);
    eccent = esol;
    mean = msol;
    ea();
    msol = ea1;
    vsol = 2 * atan(sqrt((1+esol) / (1-esol)) * tan(msol / 2));
    rsol = asol * (1-esol * cos(msol));
    xsol = rsol * cos(wsol + vsol);
    ysol = rsol * sin(wsol + vsol) * cos(oblq);
    zsol = rsol * sin(wsol + vsol) * sin(oblq);
    rasol = atan(ysol / xsol);
    if ((xsol < 0) && (ysol >= 0))
      rasol = rasol + PI;
    if ((xsol < 0) && (ysol < 0))
      rasol = rasol + PI;
    if ((xsol >= 0) && (ysol < 0))
      rasol = rasol + P2;
    rasol = rasol * (180 / PI);
    rasol = rasol / 15;
    dcsol = atan(zsol / sqrt(xsol * xsol + ysol * ysol));
    dcsol = dcsol * (180/PI);
    mag = -27;
    ra = rasol;
    dc = dcsol;
    o = "Sun";
    r = rsol;
    solrad = r;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * 331950) / ((149598770E5 * r) * (149598770E5 * r));

  }

/*** Mercury Elements Update ***/

void mercel ()

  {
    double amer, emer, imer, umer, wmer, mmer;

    amer = .3870986;
    emer = .20561421 + 2.046E-5 * t - 3E-8 * t2;
    imer = .1222233228 + 3.244777E-5 * t - 3.2E-7 * t2;
    umer = .8228519595 + .0206857877 * t + 3.0349E-6 * t2;
    wmer = 1.324699618 + .0271484026 * t + 5.1439E-6 * t2;
    mmer = 2608.787533013 * t;
    mmer = fnmodulo (mmer, P2);
    mmer = mmer + 1.7851119476;
    mmer = fnmodulo (mmer, P2);
    mmer = mmer + 1.16355E-7 * t2;
    mmer = fnmodulo (mmer, P2);
    eccent = emer;
    mean = mmer;
    ea();
    v0 = 1.16;
    ma1 = .02838;
    ma2 = .0001023;
    mmer = ea1;
    a = amer;
    e = emer;
    i = imer;
    u = umer;
    w = wmer;
    ea1 = mmer;
    radec();
    mag = mag1;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * .0553) / ((149598770E5 * r) * (149598770E5 * r));
    o = "Mercury";
  }

/*** Venus Elements Update ***/

void venuel ()

  {
    double aven, even, iven, uven, wven, mven;

    aven = .7233316;
    even = 6.82069E-3 - 4.774E-5 * t + 9.099999E-8 * t2;
    iven = .0592300268 + 1.7555E-5 * t - 1.69685E-9 * t2;
    uven = 1.32260435 + .0157053454 * t + .0000071558499 * t2;
    wven = 2.271787459 + .0245748661 * t - 1.70412E-5 * t2;
    mven = 1021.328349 * t;
    mven = fnmodulo (mven,P2);
    mven = mven + 3.7106261796;
    mven = fnmodulo (mven,P2);
    mven = mven + 2.24459E-5 * t2;
    mven = fnmodulo (mven,P2);
    eccent = even;
    mean = mven;
    ea();
    v0 = 4;
    ma = .01322;
    ma1 = 4.247E-7;
    mven = ea1;
    a = aven;
    e = even;
    i = iven;
    u = uven;
    w = wven;
    ea1 = mven;
    radec();
    mag = mag2;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * .815) / ((149598770E5 * r) * (149598770E5 * r));
    o = "Venus";
  }

/***  Mars Elements Update ***/

void marsel ()

  {
    double amar, emar, imar, umar, wmar, mmar;

    amar = 1.5236915;
    emar = 9.331289E-2 + 9.2064E-5 * t - 7.7E-8 * t2;
    imar = .0322944089 - 1.1781E-5 * t + 2.2105E-7 * t2;
    umar = .8514840374 + .0134563431 * t - .000000024240684 * t2 - 9.31E-8 * t3;
    wmar = 5.833208059 + .0321272937 * t + 2.266504E-6 * t2 - 2.0847E-08 * t3;
    mmar = 334.0535492 * t;
    mmar = fnmodulo (mmar,P2);
    mmar = mmar + 5.576840523;
    mmar = fnmodulo (mmar,P2);
    mmar = mmar + 3.1557E-6 * t2;
    mmar = fnmodulo (mmar,P2);
    mmar = mmar + .000000020846988 * t3;
    mmar = fnmodulo (mmar,P2);
    eccent = emar;
    mean = mmar;
    ea();
    v0 = -1.3;
    ma = .01486;
    mmar = ea1;
    a = amar;
    e = emar;
    i = imar;
    u = umar;
    w = wmar;
    ea1 = mmar;
    radec();
    mag = mag3;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * .1074 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Mars";
  }

/***  Jupiter Elements Update  ***/

void jupiel ()

  {
    double d2, ajup, ejup, ijup, wjup, ujup, mjup;

    d2 = julian - 2445920.5;
    ajup = 5.202743 - 6.75E-7 * d2;
    ejup = .0480453 + 1.17E-7 * d2;
    ijup = .022770962 - 5.2357E-7 * d2;
    wjup = .272271363 + 2.33874E-6 * d2;
    ujup = 1.7511935 + .0000785398144 * d2;
    mjup = .0014480674 * d2;
    mjup = fnmodulo (mjup,P2);
    mjup = mjup + 4.73761928;
    mjup = fnmodulo (mjup,P2);
    eccent = ejup;
    mean = mjup;
    ea();
    v0 = -8.93;
    mjup = ea1;
    a = ajup;
    e = ejup;
    i = ijup;
    u = ujup;
    w = wjup;
    ea1 = mjup;
    radec();
    mag = mag4;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * 317.89 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Jupiter";
  }

/***  Saturn Elements Update  ***/

void satuel ()

  {
    double d2, asat, esat, isat, usat, wsat, msat;

    d2 = julian - 2445920.5;
    asat = 9.567498 - 1.927E-5 * d2;
    esat = .0507778 - 3.8E-7 * d2;
    isat = .043364974 - 6.1083E-9 * d2;
    usat = 1.984622617 + 4.27635E-7 * d2;
    wsat = 1.640340714 - 5.427098E-5 * d2;
    msat = .00063947206 * d2;
    msat = fnmodulo (msat, P2);
    msat = msat + 2.232414341;
    msat = fnmodulo (msat, P2);
    eccent = esat;
    mean = msat;
    ea();
    v0 = -8.68;
    msat = ea1;
    a = asat;
    e = esat;
    i = isat;
    u = usat;
    w = wsat;
    ea1 = msat;
    radec();
    mag = mag4;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * 95.17 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Saturn";
  }


/***  Uranus Elements Update  ***/

void uranel ()

  {
    double d2, aura, eura, iura, uura, wura, mura;

    d2 = julian - 2445920.5;
    aura = 19.30476 - .0000388 * d2;
    eura = .04786 - 3.592E-6 * d2;
    iura = .01351670238 + 1.65809E-8 * d2;
    uura = 1.292529756 - 5.232E-8 * d2;
    wura = 3.098400987 - .000014652015 * d2;
    mura = .00022372245 * d2;
    mura = fnmodulo (mura,P2);
    mura = mura + 1.218463219;
    mura = fnmodulo (mura,P2);
    eccent = eura;
    mean = mura;
    ea();
    v0 = -6.85;
    ma = 0;
    mura = ea1;
    a = aura;
    e = eura;
    i = iura;
    u = uura;
    w = wura;
    ea1 = mura;
    radec();
    mag = mag4;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * 14.56 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Uranus";
  }

/***  Neptune Elements Update  ***/

void neptel ()

  {
    double d2, anep, enep, inep, unep, wnep, mnep;

    d2 = julian - 2445920.5;
    anep = 30.2846 + 6.05E-6 * d2;
    enep = .0063043 + 3.7765E-6 * d2;
    inep = .03088499734 - 4.27603E-8 *d2;
    unep = 2.300171164 + 1.23922E-06 * d2;
    wnep = 6.170873363 + .000047734765 * d2;
    wnep = fnmodulo (wnep, P2);
    mnep = .000063052645 * d2;
    mnep = fnmodulo (mnep, P2);
    mnep = mnep + 4.846539519;
    mnep = fnmodulo (mnep, P2);
    eccent = enep;
    mean = mnep;
    ea();
    v0 = -7.05;
    ma = 0;
    mnep = ea1;
    a = anep;
    e = enep;
    i = inep;
    u = unep;
    w = wnep;
    ea1 = mnep;
    radec();
    mag = mag4;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * 17.24 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Neptune";
  }

/***  Pluto Elements Update  ***/

void plutel ()

  {
    double d2, aplu, eplu, iplu, uplu, wplu, mplu;

    d2 = julian - 2445920.5;
    aplu = 39.74674 - .000342 * d2;
    eplu = .2539553 - 5.974E-6 * d2;
    iplu = .2989981134 - 5.7593E-8 * d2;
    uplu = 1.927194303 + 1.090865E-06 * d2;
    wplu = 3.916583414 + 1.430297E-05 * d2;
    mplu = 5.829138E-5 * d2;
    mplu = fnmodulo (mplu, P2);
    mplu = mplu + 6.150615326;
    mplu = fnmodulo (mplu, P2);
    eccent = eplu;
    mean = mplu;
    ea();
    v0 = -.97;
    mplu = ea1;
    a = aplu;
    e = eplu;
    i = iplu;
    u = uplu;
    w = wplu;
    ea1 = mplu;
    radec();
    mag = mag4;
    pull = (5.98E22 * 6.67E-8 * 5.98E27 * .02 )/ ((149598770E5 * r) * (149598770E5 * r));
    o = "Pluto";
  }

/***  ea  ***/

void ea()

  {
    double e0, dm, de;

    e0 = mean + eccent * sin(mean);
    do
      {
        dm = mean - (e0 - eccent * sin(e0));
        de = dm / (1 - eccent * cos(e0));
        e0 = e0 + de;
      }
    while (fabs(dm) > .0000001);
    ea1 = e0;
  }

/***  Generalized Solution for RA & DEC of a given Planet  ***/

void radec()

  {
    double l1, m1, n1, l2, m2, n2, p3, p4, p5, p6, b1, x, y, z, rad;
    double dis, phase;
    l1 = cos(u) * cos(w-u) - sin(u) * sin(w-u) * cos(i);
    m1 = (sin(u) * cos(w-u)) + (cos(u) * sin(w-u) * cos(i));
    n1 = sin(w-u) * sin(i);
    l2 = -cos(u) * sin(w-u) - sin(u) * cos(w-u) * cos(i);
    m2 = -sin(u) * sin(w-u) + cos(u) * cos(w-u) * cos(i);
    n2 = cos(w-u) * sin(i);
    p3 = m1 * cos(oblq) - n1 * sin(oblq);
    p4 = m1 * sin(oblq) + n1 * cos(oblq);
    p5 = m2 * cos(oblq) - n2 * sin(oblq);
    p6 = m2 * sin(oblq) + n2 * cos(oblq);
    b1 = a * sqrt (1-e*e);
    x = a*l1*cos(ea1) + b1*l2*sin(ea1) - a*e*l1;
    y = a*p3*cos(ea1) + b1*p5*sin(ea1) - a*e*p3;
    z = a*p4*cos(ea1) + b1*p6*sin(ea1) - a*e*p4;
    rad = sqrt(x*x + y*y + z*z);
    x = x + xsol;
    y = y + ysol;
    z = z + zsol;
    ra = atan(y/x);
    if (x<0 && y>=0)
      ra = ra + PI;
    if (x<0 && y<0)
      ra = ra + PI;
    if (x>0 && y<0)
      ra = ra + P2;
    ra = ra * (180 / PI);
    ra = ra / 15;
    dc = atan(z / sqrt(x*x + y*y));
    dc = dc * (180 / PI);
    r = sqrt (x*x + y*y + z*z);
    dis = r;
    phase = rad*rad + dis*dis - solrad*solrad;
    phase = phase / (2*(rad*dis));
    phase = fnacos(phase);
    phase = phase * (180 / PI);
    mag1 = v0 + 5 * (log(dis*rad) / log(10)) + ma1*(phase-50) + ma2*
           ((phase-50) * (phase-50));
    mag2 = v0 + 5 * (log(dis*rad) / log(10)) + ma*phase + ma1 *
           (phase*phase*phase);
    mag3 = v0 + 5 * (log(dis*rad) / log(10)) + ma*phase;
    mag4 = v0 + 5 * (log(dis*rad) / log(10));
  }

/*** gotoXY ***/

void gotoxy (int x, int y)
{
  struct REGPACK regs;
  regs.r_bx = 0x0000;
  regs.r_ax = 0x0200;
  regs.r_dx = ((y-1) << 8) + x-1;
  intr ('\x10', &regs);
}

/***  clear screen ***/

void clrscr ()

  {
    struct REGPACK regs;
    regs.r_ax = 0x0F00;
    intr ('\x10',&regs);
    regs.r_ax = regs.r_ax & 0x00FF;
    intr ('\x10',&regs);
  }

/*** window frame ***/

void windowfr (int xl, int yl, int xr, int yr)

  {
    int ii;
    gotoxy (xl,yl);
    printf ("\x0c9");
    for (ii = xl+1; ii <= xr-1; ii++)
      printf ("\x0cd");
    printf ("\x0bb");
    for (ii = yl+1; ii <= yr-1; ii++)
      {
        gotoxy (xl,ii);
        printf ("\x0ba");
        gotoxy (xr,ii);
        printf ("\x0ba");
      }
    gotoxy (xl,yr);
    printf ("\x0c8");
    for (ii = xl+1; ii <= xr-1; ii++)
      printf ("\x0cd");
    printf ("\x0bc");
  }

/*** Header ***/

void header ()

  {
    if (prn == 1)
      {
        /*  Print out header */
      }
    printf ("Date of Calculation: %2.0f/%2.0f/%4.0f",month,day,year);
    printf (" = Julian Date: %g\n",julian);
    printf ("At %2.0f:%2.0f:%2.0f Hours (GMT)\n\n",hour,minute,second);
    printf ("Object        Right                          Distance               Gravity\n");
    printf ("Name          Ascension       Declination    from Earth  Magnitude  Pull (cm/s)\n");
    printf ("===============================================================================\n");
  }

/*** Print line ***/

void printou()

  {
    int ra1, ra2, dc1, dc2;
    double ra3, dc3;
    ra1 = (int) ra;
    dc1 = (int) dc;
    ra = (ra - ra1) * 60;
    dc = (dc - dc1) * 60;
    ra2 = (int) fabs(ra);
    dc2 = (int) fabs(dc);
    ra = (ra - (int) ra) * 60;
    dc = (dc - (int) dc) * 60;
    ra3 = fabs(ra);
    dc3 = fabs(dc);
    printf ("%-13s%2dh%2d'%5.2f      %+3dx%2d'%5.2f  %12.9f    %+6.2f  %2.3E\n",o,ra1,ra2,ra3,dc1,dc2,dc3,r,mag,pull);
    if (prn == 1)
      {
      /*  send to printer */
      }

   }

/***  epochfig  ***/
void epochfig ()

  {
    double im,j;

    im = 12 * (year+4800) + month - 3;
    j = (2*(im - (long int)(im / 12) * 12) + 7 + 365*im) / 12;
    j = (long int) j + day + (long int) (im / 48) - 32083;
    julian = j + (long int)(im/4800) - (long int)(im/1200) + 37.5;
    julian = julian + (hour+minute/60 + second/3600)/24;
    hour = hour + minute/60 + second/3600;
    t = julian - EPOCH;
    t = t / 36525;
    t2 = t * t;
    t3 = t * t * t;
  }

/*** input dates ***/
void inpdates()

  {
    double yjt;
    int gregerr;

    do
      {
        do
          {
            gotoxy (1,1);
            printf ("Month? ");
            scanf ("%f",&month);
          }
        while (month < 0 || month > 12);
        do
          {
            gotoxy (20,1);
            printf ("  Day? ");
            scanf ("%f",&day);
          }
        while (day < 0 || day > 31);
        gotoxy (40,1);
        printf ("  Year? ");
        scanf ("%f",&year);
        yjt = year + month/100 + day/10000;
        if (yjt >= 1582.1005 && yjt < 1582.1015)
          gregerr = 1;
        else
          gregerr = 0;
        if (gregerr)
          printf ("Impossible date, try again");
      }
    while (gregerr);
    do
      {
        gotoxy (1,2);
        printf ("Hour (GMT)? ");
        scanf ("%f",&hour);
      }
    while (hour < 0 || hour > 24);
    do
      {
        gotoxy (20,2);
        printf ("Minute? ");
        scanf ("%f",&minute);
      }
    while (minute < 0 || minute > 60);
    do
      {
        gotoxy (40,2);
        printf ("Second? ");
        scanf ("%f",&second);
      }
    while (second < 0 || second > 60);
  }

/**************************************************************************/
/***********************  Ephemeris Main Program  *************************/
/**************************************************************************/

main()

  {
    double endjulian;
    int selection, planet, increment;
    float stmonth,stday,styear,sthour;

    char *yn, *another;
    another = (char *) malloc(1);
    filename = (char *) malloc(40);
    yn = (char *) malloc(1);
    increment = 1;
    selection = 1;
    planet = 0;
    do
      {
        clrscr ();
        windowfr (10,5,70,20);
        gotoxy (11,6);
        printf ("(0)  Exit to Dos");
        gotoxy (11,8);
        printf ("(1)  Display/Print single date coordinates");
        gotoxy (11,9);
        printf ("(2)  Output file of coordinates");
        gotoxy (11,19);
        printf ("Selection --->");
        scanf ("%d",&selection);
        if (selection == 1)
          {
            do
              {
                clrscr();
                inpdates();
                yn = "N";
/*              printf ("\nDo you want a printout of the results ? ");
                scanf (" %c",yn);                       */
                clrscr ();
                if (*yn == 'Y' || *yn == 'y')
                  prn = 1;
                else
                  prn = 0;
                epochfig();
                header();
                solael();
                printou();
                mercel();
                printou();
                venuel();
                printou();
                moonel();
                printou();
                marsel();
                printou();
                jupiel();
                printou();
                satuel();
                printou();
                uranel();
                printou();
                neptel();
                printou();
                plutel();
                printou();
                another = "Y";
                printf ("\n\n\n\nDo Another? ");
                scanf (" %c",another);
              }
            while (*another == 'Y' || *another == 'y');
          }
        if (selection == 2)
          {
            clrscr ();
            windowfr (10,2,70,23);
            gotoxy (11,3);
            printf ("(0) Exit");
            gotoxy (11,5);
            printf ("(1) Sun");
            gotoxy (11,6);
            printf ("(2) Mercury");
            gotoxy (11,7);
            printf ("(3) Venus");
            gotoxy (11,8);
            printf ("(4) Moon");
            gotoxy (11,9);
            printf ("(5) Mars");
            gotoxy (11,10);
            printf ("(6) Jupiter");
            gotoxy (11,11);
            printf ("(7) Saturn");
            gotoxy (11,12);
            printf ("(8) Uranus");
            gotoxy (11,13);
            printf ("(9) Neptune");
            gotoxy (11,14);
            printf ("(10) Pluto");
            gotoxy (11,17);
            printf ("Select Planet to output --->");
            scanf ("%d",&planet);
            clrscr ();
            if (planet >= 1 && planet <= 9)
              {
                gotoxy (1,10);
                printf ("Enter the starting date & times");
                inpdates();
                stmonth = month;
                stday = day;
                styear = year;
                sthour = hour;
                clrscr();
                gotoxy (1,10);
                printf ("Enter the ending date & times");
                inpdates();
                epochfig();
                endjulian = julian;
                gotoxy (1,12);
                printf ("(1) Daily  (2) Weekly  (3) Monthly  (4) Yearly");
                scanf ("%d",&increment);
                gotoxy (1,14);
                printf ("Enter the output file name: ");
                scanf ("%s",filename);
                outfile = fopen(filename,"w");
                if (outfile == NULL)
                  {
                    printf ("\n Error opening file %s",filename);
                  }
                else
                  {
                    fprintf (outfile, "right ascension,declination,distance from earth,magnitude,julian date,gravity pull\n");
                    fprintf (outfile, "For the planet: %d. From %2.0f/%2.0f/%4.0f to %2.0f/%2.0f/%4.0f\n",
                        planet,stmonth,stday,styear,month,day,year);
                    fprintf (outfile,"1=Sun,2=Mercury,3=Venus,4=Moon,5=Mars,6=Jupiter,7=Saturn\n");
                    fprintf (outfile,"8=Uranus,9=Neptune,10=Pluto");
                    clrscr();
                    month = stmonth;
                    day = stday;
                    year = styear;
                    hour = sthour;
                    epochfig();
                    header();
                    do
                      {
                        t = julian - EPOCH;
                        t = t / 36525;
                        t2 = t * t;
                        t3 = t * t * t;
                        solael();
                        switch (planet)
                          {
                            case 1: solael();
                                    break;
                            case 2: mercel();
                                    break;
                            case 3: venuel();
                                    break;
                            case 4: moonel();
                                    break;
                            case 5: marsel();
                                    break;
                            case 6: jupiel();
                                    break;
                            case 7: satuel();
                                    break;
                            case 8: uranel();
                                    break;
                            case 9: neptel();
                                    break;
                            case 10: plutel();
                                    break;
                          }
                        printou();
                        fprintf(outfile,"%g,%g,%g,%g,%g,%g\n",ra,dc,r,mag,julian,pull);
                        switch (increment)
                          {
                            case 1: julian = julian + 1;
                                    break;
                            case 2: julian = julian + 7;
                                    break;
                            case 3:
                                month = month + 1;
                                if (month > 12)
                                  {
                                    month = 1;
                                    year = year + 1;
                                  }
                                epochfig();
                                break;
                            case 4:
                                year = year + 1;
                                epochfig();
                                break;
                          }
                      }
                    while (julian <= endjulian);
                    fclose (outfile);
                  }
              }
          }
      }
    while (selection != 0);
  }