/* almanac.c

   Version 3.2 for OS9 and OS9/68000

   By John M. Semler [73107,507]

   Syntax of command:
       "almanac [<longitude> <latitude>] <date> <time> [<timezone>]"
   where
       "<longitude>" is local longitude (DDD.MM,+W,-E)
       "<latitude>" is local latitude (DD.MM,+N,-S)
       "<date>" is calendar date (MMDD.YYYY)
       "<time>" is time (24 hour format,HH.MMSS)
       "<timezone>" is one of the following: "EST","CST","MST", or "PST"

   This program will calculate the right ascension and declination of
   the Sun, Moon, and the eight major planets given the calendar date and
   time.  If the observer's longitude and latitude was given, it will also
   calculate the azimuth and altitude for the Sun, Moon, and each of the
   planets.  Most of the formulas and constants was typed in from "The
   Astronomical Almanac 1990" and will need to be revised every year
   to maintain accuracy.  Some of the formulas used was also taken from
   "Methods of Celestial Mechanics" by Brouwer and Clemence.

   This program takes into account precession, nutation, barycentric
   position of the Earth-Moon system, and planetary aberration when
   calculating the right ascension and declination (geocentric) of the
   Sun, and planets.

   This program will also take into account the effects of diurnal
   aberration, and diurnal parallax in its calculation of right ascension
   and declination (topocentric) for the Sun, and planets when given the
   longitude and latitude of the place of observation.  Diurnal aberration
   contributes no more than 0.319" displacement.

   Right ascension, declination, and distance for the Moon is corrected for
   topocentric observation (diurnal parallax only) if given the observer's
   longitude and latitude.  Diurnal aberration is ignored.

   Atmospheric refraction is ignored when calculating the azimuth and
   altitude for the Sun, Moon, and planets.

   The two sets of orbital elements as used by this program should give
   good coverage for the year 1990.  The orbital elements seems to work
   well for dates within 100 days of the date of epoch for each of the
   set of elements.  There will be a small discontinuous jump in the
   positions of the Sun, and planets close to the Sun for a date exactly
   between the two date of epoch (5/09/1990 at 0:00:00).  See the
   comparison of "almanac.c" and "The  Astronomical Almanac" for a
   demonstration of accuracy.

   The formulas used to compute the geocentric coordinates of the Moon is
   accurate to 0.3 degrees in right ascension and 0.2 degrees in declination.

   To compile and link under OS9/6809, you will need the C math enhanced
   library "CLIBT.L" found in data library 3 of the OS9 forum.

 *****************************************************************************

   Sample problem:

   Given that my longitude is 90 deg 30 min west, and my latitude is
   38 deg 45 min north on December 2, 1989 at 19 hour 5 min central
   standard time, where must I look to find the planets?

   Enter the command as follows:

   OS9:almanac 90.30 38.45 1202.1989 19.05 cst

   Topocentric Coordinates      25:05:00      12/02/1989  |  Longitude   90.30
                                                          |  Latitude    38.45
   Object        R.A.       Dec.      Diam.       A.U.    |  Azimuth  Altitude
                                                          |
   Sun         16.37025   -22.0430   0.322691    0.98579  |   262.50    -27.33
   Moon        20.24---   -21.14--   0.3055--    0.00258  |   228.31     13.02
   Mercury     17.29203   -25.1844   0.000495    1.35639  |   252.38    -19.24
   Venus       19.50284   -23.5233   0.003437    0.48523  |   232.45      5.56
   Mars        15.08371   -17.2131   0.000380    2.46025  |   282.02    -42.02
   Jupiter      6.38425    22.5737   0.004633    4.24949  |    64.44      5.23
   Saturn      18.53211   -22.3231   0.001524   10.85681  |   242.51     -2.33
   Uranus      18.17395   -23.3852   0.000345   20.27702  |   247.17     -9.34
   Neptune     18.47235   -22.0813   0.000216   31.05958  |   244.04     -3.19
   Pluto       15.11527    -2.0714   0.000014   30.51682  |   295.41    -31.21

   Provided that there are no clouds blocking my view, I would see Jupiter
   rising in the east-northeast and Venus setting in the southwest 4 deg
   to the right of and 7 deg below the Moon.

 *****************************************************************************

   Comparison of "almanac.c" calculations with "The Astronomical Almanac"


    Geocentric Coordinates        0:00:00       1/01/1990

    Object        R.A.       Dec.      Diam.       A.U.

    Sun         18.44501   -23.0232   0.323176    0.98334 (almanac.c)
                18.44500   -23.0232   0.323182    0.98334 (1990 Almanac)

    Mercury     19.50448   -20.3120   0.000859    0.78212
                19.50448   -20.3120   0.000860    0.78212

    Venus       20.32486   -16.5853   0.005422    0.30764
                20.32485   -16.5854   0.005424    0.30763

    Mars        16.31555   -21.5538   0.000405    2.31050
                16.31555   -21.5538   0.000405    2.31050

    Jupiter      6.22426    23.1317   0.004721    4.17058
                 6.22426    23.1317   0.004721    4.17058

    Saturn      19.07330   -22.1419   0.001503   11.00586
                19.07330   -22.1419   0.001503   11.00586

    Uranus      18.25077   -23.3513   0.000344   20.36012
                18.25077   -23.3513   0.000344   20.36012

    Neptune     18.51558   -22.0317   0.000215   31.19316
                18.51558   -22.0317   0.000215   31.19317


    Geocentric Coordinates        0:00:00       4/01/1990

    Object        R.A.       Dec.      Diam.       A.U.

    Sun          0.40280     4.2120   0.320073    0.99922 (almanac.c)
                 0.40280     4.2121               0.99921 (1990 Almanac)

    Mercury      1.27392     9.5637   0.000567    1.18452
                 1.27392     9.5637               1.18452

    Venus       21.45372   -12.0642   0.002401    0.69473
                21.45372   -12.0641               0.69471

    Mars        21.11460   -17.2559   0.000540    1.73292
                21.11459   -17.2559               1.73292

    Jupiter      6.12019    23.2923   0.003764    5.23153
                 6.12019    23.2923               5.23153

    Saturn      19.45073   -21.0229   0.001621   10.20436
                19.45073   -21.0229              10.20437

    Uranus      18.41311   -23.2351   0.000362   19.34436
                18.41311   -23.2351              19.34437

    Neptune     19.02363   -21.4754   0.000221   30.25279
                19.02363   -21.4754              30.25280


    Geocentric Coordinates        0:00:00       7/01/1990

    Object        R.A.       Dec.      Diam.       A.U.

    Sun          6.38504    23.0830   0.312785    1.01663 (almanac.c)
                 6.38504    23.0830               1.01663 (1990 Almanac)

    Mercury      6.29541    24.2114   0.000508    1.32169
                 6.29540    24.2113               1.32169

    Venus        4.22126    19.5916   0.001234    1.35146
                 4.22126    19.5916               1.35147

    Mars         1.24315     6.4034   0.000784    1.19341
                 1.24315     6.4034               1.19341

    Jupiter      7.23423    22.1530   0.003169    6.21322
                 7.23423    22.1530               6.21322

    Saturn      19.39200   -21.2302   0.001834    9.02256
                19.39199   -21.2302               9.02256

    Uranus      18.32562   -23.3311   0.000381   18.39602
                18.32562   -23.3312              18.39603

    Neptune     18.57281   -21.5418   0.000230   29.19258
                18.57281   -21.5419              29.19259


    Geocentric Coordinates        0:00:00      10/01/1990

    Object        R.A.       Dec.      Diam.       A.U.

    Sun         12.27469    -3.0005   0.315686    1.00124 (almanac.c)
                12.27469    -3.0005               1.00124 (1990 Almanac)

    Mercury     11.33473     4.4728   0.000592    1.13579
                11.33473     4.4728               1.13578

    Venus       11.59577     1.3423   0.000984    1.69538
                11.59576     1.3423               1.69539

    Mars         4.42112    21.0737   0.001366    0.68539
                 4.42113    21.0737               0.68539

    Jupiter      8.43191    18.3334   0.003460    5.69023
                 8.43191    18.3334               5.69023

    Saturn      19.21155   -22.0955   0.001696    9.75936
                19.21155   -22.0955               9.75937

    Uranus      18.24572   -23.3827   0.000360   19.43533
                18.24572   -23.3827              19.43533

    Neptune     18.51028   -22.0512   0.000223   30.11359
                18.51028   -22.0512              30.11360


    Geocentric Coordinates        0:00:00      12/31/1990

    Object        R.A.       Dec.      Diam.       A.U.

    Sun         18.39198   -23.0801   0.323177    0.98333 (almanac.c)
                18.39200   -23.0801               0.98331 (1990 Almanac)

    Mercury     17.38050   -20.1131   0.000920    0.73068
                17.38051   -20.1130               0.73066

    Venus       19.42131   -22.3836   0.001019    1.63744
                19.42131   -22.3836               1.63743

    Mars         3.39446    21.5558   0.001395    0.67113
                 3.39448    21.5560               0.67110

    Jupiter      8.58576    17.4930   0.004444    4.42958
                 8.58575    17.4930               4.42959

    Saturn      19.50112   -21.1024   0.001514   10.92994
                19.50112   -21.1024              10.93001

    Uranus      18.42111   -23.2426   0.000343   20.42806
                18.42112   -23.2426              20.42812

    Neptune     19.00478   -21.5440   0.000215   31.18146
                19.00478   -21.5440              31.18152


    Geocentric Coordinates        0:00:00       1/01/1985

    Object        R.A.       Dec.      Diam.       A.U.

    Sun         18.45469   -23.0129   0.323183    0.98330 (almanac.c)
                18.45451   -23.0134               0.98325 (1985 Almanac)

    Mercury     17.08359   -20.4224   0.000699    0.96193
                17.08332   -20.4229               0.96226

    Venus       21.56379   -14.1232   0.001981    0.84208
                21.56344   -14.1303               0.84289

    Mars        22.29164   -10.3117   0.000544    1.71964
                22.29182   -10.3111               1.71959

    Jupiter     19.32578   -21.5842   0.003218    6.11848
                19.32600   -21.5847               6.11755

    Saturn      15.31387   -16.5632   0.001568   10.55455
                15.31357   -16.5627              10.57002

    Uranus      16.56332   -22.3839   0.000351   19.92796
                16.56270   -22.3826              19.93770

    Neptune     18.06306   -22.1917   0.000215   31.21953
                18.06252   -22.1917              31.22774

 ****************************************************************************/

#define V  173.1421288    /* A.U. per julian day (speed of light) */
#define ER2AU (6378160.0/149600.0e6) /* A.U. per Earth radii */
#define PI 3.141592653589796

#define Deg(x) (((double)(x))/PI*180.0)
#define Rad(x) (((double)(x))/180.0*PI)
#define sqr(x) ((x)*(x))
#define sgn(x) ((x) < 0.0 ? -1.0: 1.0)
#define angle(x,y) (atan((x)/(y))+((y)<=0.0?PI:(x)<=0.0?2*PI:0.0));

typedef struct {
    double i;   /* Inclination */
    double ohm; /* Longitude of ascending node */
    double nw;  /* Longitude of perhelion */
    double a;   /* Mean distance */
    double n;   /* Daily motion */
    double e;   /* Eccentricity */
    double L;   /* Mean longitude */
} elementtype;

/* The following elements have been typed in from page E3 of "The
   Astronomical Almanac 1990".  The orbital elements is for JD 2447920.5
   and JD 2448120.5 referred to the mean ecliptic and equinox of J2000.0 */

elementtype element[][9] = {

    {
        { 1.3200000e-3,   3.5240000e+2,   1.0290260e+2,   1.0000029e+0,
          9.8560490e-1,   1.6700600e-2,   1.2812283e+2 },

        { 7.0055100e+0,   4.8343500e+1,   7.7442900e+1,   3.8709700e-1,
          4.0923650e+0,   2.0564000e-1,   1.7956645e+2 },

        { 3.3947200e+0,   7.6709300e+1,   1.3145300e+2,   7.2334530e-1,
          1.6020870e+0,   6.7801000e-3,   1.3505681e+2 },

        { 1.8504000e+0,   4.9588800e+1,   3.3596600e+2,   1.5236842e+0,
          5.2403670e-1,   9.3459700e-2,   2.5608918e+2 },

        { 1.3046400e+0,   1.0046940e+2,   1.5709300e+1,   5.2033720e+0,
          8.3077600e-2,   4.8207300e-2,   9.3230120e+1 },

        { 2.4872900e+0,   1.1367870e+2,   9.2824700e+1,   9.5161770e+0,
          3.3579350e-2,   5.5383700e-2,   2.8872988e+2 },

        { 7.7209000e-1,   7.4011500e+1,   1.6872880e+2,   1.9154770e+1,
          1.1757050e-2,   4.7564700e-2,   2.7073261e+2 },

        { 1.7706200e+0,   1.3177670e+2,   5.5806000e+1,   2.9999730e+1,
          5.9984530e-3,   1.0263400e-2,   2.8323073e+2 },

        { 1.7150490e+1,   1.1022840e+2,   2.2375390e+2,   3.9442420e+1,
          3.9788610e-3,   2.4812230e-1,   2.2438521e+2 }
    },

    {
        { 1.2600000e-3,   3.5370000e+2,   1.0300050e+2,   9.9999980e-1,
          9.8560940e-1,   1.6674400e-2,   3.2524544e+2 },

        { 7.0054700e+0,   4.8342900e+1,   7.7443400e+1,   3.8709920e-1,
          4.0923310e+0,   2.0562730e-1,   2.7803485e+2 },

        { 3.3946900e+0,   7.6707300e+1,   1.3129300e+2,   7.2333200e-1,
          1.6021310e+0,   6.7721000e-3,   9.5483840e+1 },

        { 1.8503700e+0,   4.9586300e+1,   3.3594400e+2,   1.5236583e+0,
          5.2405010e-1,   9.3408900e-2,   8.9231000e-1 },

        { 1.3046600e+0,   1.0046950e+2,   1.5683700e+1,   5.2032640e+0,
          8.3080190e-2,   4.8228700e-2,   1.0984326e+2 },

        { 2.4874000e+0,   1.1367850e+2,   9.3140800e+1,   9.5160000e+0,
          3.3580290e-2,   5.5299300e-2,   2.9540811e+2 },

        { 7.7191000e-1,   7.4005800e+1,   1.6912770e+2,   1.9159600e+1,
          1.1752600e-2,   4.7872700e-2,   2.7303858e+2 },

        { 1.7710400e+0,   1.3176910e+2,   5.7394000e+1,   3.0003410e+1,
          5.9973510e-3,   9.7821000e-3,   2.8436803e+2 },

        { 1.7150570e+1,   1.1022780e+2,   2.2370330e+2,   3.9503410e+1,
          3.9696500e-3,   2.4928760e-1,   2.2515605e+2 }
    }
};

char *name[] = { "Sun", "Mercury", "Venus", "Mars", "Jupiter",
                 "Saturn", "Uranus", "Neptune", "Pluto" };

/* Equatorial semidiameters at unit distance for the Sun and planets (page
   E43 of "The Astronomical Almanac 1990") */

double sd[] = {0.155962, 0.000336, 0.000834, 0.000468, 0.009844,
               0.008273, 0.003502, 0.003350, 0.000207 };

/* Date of epoch for the orbital elements (Julian Date)  */

double eDate[2] = { 2447920.5, 2448120.5 };

/* obliquity for date of epoch J2000.0 */

double ee = 23.439291;

/* Math library function */

double sin(), cos(), tan(), asin(), atan(), sqrt(), atof();

/* "strucmp" is a version of "strcmp" that ignores the case of letters */

char *strucmp(s, t)
char s[], t[];
{
    int i;

    i = 0;
    while (toupper(s[i]) == toupper(t[i]))
        if (s[i++] == '\0')
            return 0;
    return s[i] - t[i];
}

/* "DD" converts DD.MMSSsss to DD.ddddd */

double DD(p)
double p;
{
    double sign;

    p = (p * (sign = sgn(p)))*100.0;
    p = (5.0/3.0*p-2.0/3.0*(int)p)/100.0;
    p = (5.0/3.0*p-2.0/3.0*(int)p)*sign;

    return p;
}

/* "DMS" converts DD.ddddd to DD.MMSSsss */

double DMS(p)
double p;
{
    double sign;

    p = (p * (sign = sgn(p)));
    p = (.6*p+.4*(int)p)*100.0;
    p = (.6*p+.4*(int)p)/100.0*sign;

    return p;
}

/* "DM" converts DD.ddddd to DD.MMmmm */

double DM(p)
double p;
{
    double sign;

    p = (p * (sign = sgn(p)));
    p = (.6*p+.4*(int)p)*sign;

    return p;
}

/* "Distance" computes distance between two points */

double Distance(cord1, cord2)
double cord1[3], cord2[3];
{
    return sqrt((cord1[0]-cord2[0])*(cord1[0]-cord2[0])+
                (cord1[1]-cord2[1])*(cord1[1]-cord2[1])+
                (cord1[2]-cord2[2])*(cord1[2]-cord2[2]));
}

/* "Rotate" rotates planetary coordinates given the rotation matrix */

Rotate(cord, matrix)
double cord[3], matrix[3][3];
{
    double newcord[3];
    int i,j;

    for (i=0; i<3; i++) {
        newcord[i]=0;
        for (j=0;j<3;j++)
            newcord[i]=newcord[i]+cord[j]*matrix[j][i];
    }
    for (i=0;i<3;i++)
        cord[i]=newcord[i];
}

/* "Gregory" converts calendar date to julian day number */

double Gregory(month, day, year)
long month, day, year;
{
    static int MDays[2][12] = {
        {0,31,59,90,120,151,181,212,243,273,304,334},
        {0,31,60,91,121,152,182,213,244,274,305,335}
    };

    long YDays, YDays1;

    YDays1 = 365*year+year/4-year/100+year/400;
    year--;
    YDays = 365*year+year/4-year/100+year/400;
    return YDays+MDays[(YDays1-YDays)==366?1:0][month-1]+day+1721424.5;
}

/* The precession matrix formula can be found on page B18 of "The
   Astronomical Almanac 1990" */

Precession(jDate, p)
double jDate, p[3][3];
{
    double T, zeta, z, theta;

    T=(jDate-2451545.0)/36525.0;
    zeta =  (( .0000050*T+.0000839)*T+.6406161)*T;
    z =     (( .0000051*T+.0003041)*T+.6406161)*T;
    theta = ((-.0000116*T-.0001185)*T+.5567530)*T;

    zeta =  Rad(zeta);
    z =     Rad(z);
    theta = Rad(theta);

    p[0][0] =  cos(zeta)*cos(theta)*cos(z)-sin(zeta)*sin(z);
    p[1][0] = -sin(zeta)*cos(theta)*cos(z)-cos(zeta)*sin(z);
    p[2][0] = -sin(theta)*cos(z);
    p[0][1] =  cos(zeta)*cos(theta)*sin(z)+sin(zeta)*cos(z);
    p[1][1] = -sin(zeta)*cos(theta)*sin(z)+cos(zeta)*cos(z);
    p[2][1] = -sin(theta)*sin(z);
    p[0][2] =  cos(zeta)*sin(theta);
    p[1][2] = -sin(zeta)*sin(theta);
    p[2][2] =  cos(theta);
}

/* The low precision nutation formula can be found on page B20 of "The
   Astronomical Almanac 1990" */

Nutation(jDate, RA, DEC)
double jDate, *RA, *DEC;
{
    double a, b, d, gamma, epsilon, ee;

    d = jDate-2447891.5;
    gamma   = -0.0048*sin(Rad(318.5-0.053*d))
              -0.0004*sin(Rad(198.8+1.971*d));
    epsilon =  0.0026*cos(Rad(318.5-0.053*d))
              +0.0002*cos(Rad(198.8+1.971*d));

    ee = Rad(23.44);
    a = (cos(ee)+sin(ee)*sin(*RA)*tan(*DEC))*gamma-cos(*RA)*tan(*DEC)*epsilon;
    b = sin(ee)*cos(*RA)*gamma+sin(*RA)*epsilon;

    *RA =  *RA  + Rad(a);
    *RA = angle(sin(*RA),cos(*RA));
    *DEC = *DEC + Rad(b);
}

/* "Barycentric" calculates the true coordinates of the Earth given the
   barycentric coordinates of the Earth-Moon system.  This formula can
   be found on page E2 of "The Astronomical Almanac 1990".  Velocity
   equations was found by taking the derivatives of the coordinates
   equations */

Barycentric(jDate, earth, v_earth)
double jDate, earth[3], v_earth[3];
{
    double T, L, dT, dL;

    T = (jDate-2451545.0)/36525.0;
    dT = 1/36525.0;
    L = Rad(218.0 + 481268.0*T);
    dL = Rad(481268.0*dT);

    earth[0] = earth[0] - 0.0000312*cos(L);
    earth[1] = earth[1] - 0.0000312*sin(L);

    v_earth[0] = v_earth[0] + 0.0000312*sin(L)*dL;
    v_earth[1] = v_earth[1] - 0.0000312*cos(L)*dL;
}

/* "Topocentric" calculates the true coordinates and velocity of the
   observer given the longitude, latitude, and the ecliptic coordinates
   of the Earth.  This formula is a version of that found on page B61 of
   "The Astronomical Almanac 1990" (rho equals unity, geocentric latitude
   equals geodetic latitude) */

Topocentric(LONG, LAT, jDate, jTime, earth, v_earth)
double LONG, LAT, jDate, jTime, earth[3], v_earth[3];
{
    double GMST, dGMST, LST, dLST, x, y, z, dx, dy, dz;

    /* Calculate Greenwich mean sidereal time and its derivative */

    GMST = 6.6265313+0.0657098243*(jDate-jTime-2447891.5)+1.00273791*24.0*jTime;
    GMST = Rad(GMST*15.0);
    dGMST = 1.00273791*24.0;
    dGMST = Rad(dGMST*15.0);

    /* Calculate local sidereal time and its derivative */

    LST = GMST-LONG;
    dLST = dGMST;

    /* Calculate geocentric coordinates and velocity */

    x = ER2AU*cos(LAT)*cos(LST);
    y = ER2AU*cos(LAT)*sin(LST);
    z = ER2AU*sin(LAT);

    dx = -ER2AU*cos(LAT)*sin(LST)*dLST;
    dy =  ER2AU*cos(LAT)*cos(LST)*dLST;
    dz =  0;

    /* Calculate new ecliptic coordinates and velocity */

    earth[0] = earth[0]+x;
    earth[1] = earth[1]+y*cos(Rad(ee))+z*sin(Rad(ee));
    earth[2] = earth[2]+z*cos(Rad(ee))-y*sin(Rad(ee));

    v_earth[0] = v_earth[0]+dx;
    v_earth[1] = v_earth[1]+dy*cos(Rad(ee))+dz*sin(Rad(ee));
    v_earth[2] = v_earth[2]+dz*cos(Rad(ee))-dy*sin(Rad(ee));
}

/* The low precision formulas for calculating the geocentric coordinates
   of the Moon can be found on page D46 of "The Astronomical Almanac 1990".
   These formulas are accurate to a precision of 0.3 degrees in right
   ascension and 0.2 degrees in declination */

Moon(jDate,RA,DEC,AU)
double jDate, *RA, *DEC, *AU;
{
    double T, lambda, beta, pie, l, m, n;

    T = (jDate - 2451545.0)/36525.0;

    lambda = 218.32+481267.883*T
           + 6.29*sin(Rad(134.9+477198.85*T))
           - 1.27*sin(Rad(259.2-413335.38*T))
           + 0.66*sin(Rad(235.7+890534.23*T))
           + 0.21*sin(Rad(269.9+954397.70*T))
           - 0.19*sin(Rad(357.5+35999.05*T))
           - 0.11*sin(Rad(186.6+966404.05*T));

    beta = 5.13*sin(Rad(93.3+483202.03*T))
         + 0.28*sin(Rad(228.2+960400.87*T))
         - 0.28*sin(Rad(318.3+6003.18*T))
         - 0.17*sin(Rad(217.6-407332.20*T));

    pie = 0.9508
        + 0.0518*cos(Rad(134.9+477198.85*T))
        + 0.0095*cos(Rad(259.2-413335.38*T))
        + 0.0078*cos(Rad(235.7+890534.23*T))
        + 0.0028*cos(Rad(269.9+954397.70*T));

    lambda = Rad(lambda);
    beta = Rad(beta);
    pie = Rad(pie);

    l = cos(beta)*cos(lambda);
    m = 0.9175*cos(beta)*sin(lambda)-0.3978*sin(beta);
    n = 0.3978*cos(beta)*sin(lambda)+0.9175*sin(beta);

    *AU = 1/sin(pie)*ER2AU;
    *RA = angle(m,l);
    *DEC = asin(n);
}

/* The low precision formulas for calculating the topocentric coordinates
   of the Moon can be found on page D46 of "The Astronomical Almanac 1990".
   These formulas corrects for displacement (diurnal parallax) due to the
   observer's topocentric location */

Moon1(RA,DEC,AU,LONG,LAT,jDate,jTime)
double *RA, *DEC, *AU, LONG, LAT, jDate, jTime;
{
    double GMST, LST, r, x, y, z;

    /* Calculate Greenwich mean sidereal time */

    GMST = 6.6265313+0.0657098243*(jDate-jTime-2447891.5)+1.00273791*24.0*jTime;
    GMST = Rad(GMST*15);

    /* Calculate local sidereal time */

    LST = GMST-LONG;

    /* Calculate new geocentric rectangular coordinates */

    r = *AU/ER2AU;
    x = r*cos(*DEC)*cos(*RA)-cos(LAT)*cos(LST);
    y = r*cos(*DEC)*sin(*RA)-cos(LAT)*sin(LST);
    z = r*sin(*DEC)-sin(LAT);
    r = sqrt(x*x+y*y+z*z);

    /* Calculate new geocentric right ascension, declination, and distance */

    *RA = angle(y,x);
    *DEC = asin(z/r);
    *AU = r*ER2AU;
}

/* "Ecliptic" calculates the ecliptic coordinates and velocity of a planet
   given the julian day number, the date of epoch, and the orbital elements.
   Velocity equations was found by taking the derivatives of the coordinates
   equations */

Ecliptic(jDate, index, planet, v_planet)
double jDate, planet[3], v_planet[3];
int index;
{
    double i,  ohm,  nw,  a,  n,  e,  L,
           M, u, f, r, dM, du, df, dr,
           LDate, aa, bb, cc, dd, ee, ff;

    if (jDate <= eDate[0]+100) {
        i   = Rad(element[0][index].i);
        ohm = Rad(element[0][index].ohm);
        nw  = Rad(element[0][index].nw);
        n   = Rad(element[0][index].n);
        L   = Rad(element[0][index].L);
        a   = element[0][index].a;
        e   = element[0][index].e;
        LDate = eDate[0];
    }
    else {
        i   = Rad(element[1][index].i);
        ohm = Rad(element[1][index].ohm);
        nw  = Rad(element[1][index].nw);
        n   = Rad(element[1][index].n);
        L   = Rad(element[1][index].L);
        a   = element[1][index].a;
        e   = element[1][index].e;
        LDate = eDate[1];
    }

    /* Calculate mean anomaly and its derivative */

    M  = (jDate-LDate)*n+L-nw; dM = n;

    /* Calculate eccentric anomaly and its derivative */

    u = M;
    u = (sin(u)*e+M-u)/(1-e*cos(u))+u;
    u = (sin(u)*e+M-u)/(1-e*cos(u))+u;
    u = (sin(u)*e+M-u)/(1-e*cos(u))+u;
    u = (sin(u)*e+M-u)/(1-e*cos(u))+u;
    du = dM/(1-e*cos(u));

    /* Calculate true anomaly and its derivative */

    f = 2*atan(sqrt((1+e)/(1-e))*tan(u/2));
    df = sqrt((1+e)/(1-e))/(1+sqr(sqrt((1+e)/(1-e))*tan(u/2)))/sqr(cos(u/2))*du;

    /* Calculate radius vector and its derivative */

    r = a*(1-e*e)/(1+e*cos(f));
    dr = a*(1-e*e)*e*sin(f)/sqr(1+e*cos(f))*df;

    /* Calculate ecliptic coordinates and velocity */

    aa = cos(f+nw-ohm);
    bb = sin(f+nw-ohm);
    cc = cos(ohm);
    dd = sin(ohm);
    ee = cos(i);
    ff = sin(i);

    planet[0] = r*(aa*cc-bb*ee*dd);
    planet[1] = r*(aa*dd+bb*ee*cc);
    planet[2] = r*bb*ff;

    v_planet[0] = dr*(aa*cc-bb*ee*dd)-r*(bb*cc+aa*ee*dd)*df;
    v_planet[1] = dr*(aa*dd+bb*ee*cc)-r*(bb*dd-aa*ee*cc)*df;
    v_planet[2] = dr*bb*ff+r*aa*ff*df;
}

/* "Heliocentric" calculates the heliocentric coordinates given the ecliptic
   coordinates */

Heliocentric(h_planet, planet)
double h_planet[3],  planet[3];
{
    h_planet[0] = planet[0];
    h_planet[1] = planet[1]*cos(Rad(ee))-planet[2]*sin(Rad(ee));
    h_planet[2] = planet[2]*cos(Rad(ee))+planet[1]*sin(Rad(ee));
}

/* "RaDec" calculates the right ascension and declination of a planet
   given the Earthly and Planetary heliocentric coordinates */

RaDec(earth, planet, RA, DEC)
double earth[3], planet[3], *RA, *DEC;
{
    *DEC = asin((planet[2]-earth[2])/Distance(planet,earth));
    *RA = angle(planet[1]-earth[1],planet[0]-earth[0]);
}

/* "AltAz" calculates the azimuth and altitude of a planet given the right
   ascension and declination of the planet, the longitude and latitude of
   the observer, and the observation time. These formulas can be found on
   page B6, B61, and B62 of "The Astronomical Almanac 1990" */

AltAz(RA, DEC, LONG, LAT, jDate, jTime, AZ, ALT)
double RA, DEC, LONG, LAT, jDate, jTime, *AZ, *ALT;
{
    double GMST, h, a, b, c;

    /* Calculate Greenwich mean sidereal time */

    GMST = 6.6265313+0.0657098243*(jDate-jTime-2447891.5)+1.00273791*24.0*jTime;
    GMST = Rad(GMST*15);

    /* Calculate local hour angle */

    h = GMST-LONG-RA;

    /* Calculate azimuth and altitude */

    a = -cos(DEC)*sin(h);
    b = sin(DEC)*cos(LAT)-cos(DEC)*cos(h)*sin(LAT);
    c = sin(DEC)*sin(LAT)+cos(DEC)*cos(h)*cos(LAT);

    *ALT = asin(c);
    *AZ = angle(a,b);
}

/* "Aberration" corrects for planetary aberration.  This formula can be
   found on page 192 of "Methods of Celestial Mechanics" */

Aberration(planet1, v_planet, v_earth, planet, earth)
double planet1[3], v_planet[3], v_earth[3], planet[3], earth[3];
{
    double tau;

    tau = Distance(earth, planet) / V;
    planet1[0] = (v_earth[0]-v_planet[0])*tau + planet[0];
    planet1[1] = (v_earth[1]-v_planet[1])*tau + planet[1];
    planet1[2] = (v_earth[2]-v_planet[2])*tau + planet[2];
}

main(argc, argv)
int argc;
char *argv[];
{
    int k;

    long month, day, year;
    long hour, min, sec;

    char *TimeZone;     /* Time zone flag */

    double LONG, LAT;   /* Observer's coordinates */
    double RA, DEC, AU; /* Equatorial coordinates */
    double AZ, ALT;     /* Local coordinates */

    double gDate;       /* Gregorian calendar date */
    double gTime;       /* Gregorian wall time */
    double jDate;       /* Julian calendar date */
    double jTime;       /* Julian fractional day time */
    double p[3][3];     /* Precession matrix for calendar date */

    double earth[3];    /* Earthly ecliptic coordinates */
    double v_earth[3];  /* Earthly ecliptic velocity */
    double h_earth[3];  /* Earthly heliocentric coordinates */

    double object[3];   /* Planetary or Solar ecliptic coordinates */
    double v_object[3]; /* Planetary or Solar ecliptic velocity */
    double h_object[3]; /* Planetary or Solar heliocentric coordinates */

    pffinit();  /* OS9/6809 needs this for printing of doubles */

    if (argc == 3 || argc == 4) {
        gDate = atof(argv[1])+0.00001;
        gTime = atof(argv[2])+0.00001;
        if (argc == 4) TimeZone = argv[3];
    }
    else if (argc == 5 || argc == 6) {
        LONG = Rad(DD(atof(argv[1])+0.00001*sgn(atof(argv[1]))));
        LAT = Rad(DD(atof(argv[2])+0.00001*sgn(atof(argv[2]))));
        gDate = atof(argv[3])+0.00001;
        gTime = atof(argv[4])+0.00001;
        if (argc == 6) TimeZone = argv[5];
    }
    else {
        printf("\n%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
"Syntax of command:\n",
"    \"almanac [<longitude> <latitude>] <date> <time> [<timezone>]\"\n",
"where\n",
"    \"<longitude>\" is local longitude (DDD.MM,+W,-E)\n",
"    \"<latitude>\" is local latitude (DD.MM,+N,-S)\n",
"    \"<date>\" is calendar date (MMDD.YYYY)\n",
"    \"<time>\" is time (24 hour format,HH.MMSS)\n",
"    \"<timezone>\" is one of the following: \"EST\", \"CST\", \"MST\", or \"PST\"\n\n",
"Azimuth and altitude is not computed if the parameter \"<longitude>\" and\n",
"\"<latitude>\" was omitted.  Universal time is assumed unless a \"<timezone>\"\n",
"flag was supplied.  For the summer months, enter your time minus 1 hour if\n",
"daylight saving time is in effect in your local time zone.\n\n",
"All displayed angles are in degrees, minutes, seconds (if displayed) format\n",
"except for right ascension which is in hours, minutes, seconds format.\n",
"Azimuth is measured clockwise from north in the plane of the horizon.\n");
        exit(0);
    }

    if (strucmp(TimeZone,"EST") == 0)
        gTime = gTime + 5.0;
    else if (strucmp(TimeZone,"CST") == 0)
        gTime = gTime + 6.0;
    else if (strucmp(TimeZone,"MST") == 0)
        gTime = gTime + 7.0;
    else if (strucmp(TimeZone,"PST") == 0)
        gTime = gTime + 8.0;

    month = (long)(gDate/100);
    day = (long)(gDate)-100*month;
    year = (gDate-(long)(gDate))*10000;
    hour = (long) gTime;
    min = (long)((gTime-hour)*100);
    sec = (long)(((gTime-hour)*100-min)*100);

    jTime = (hour+min/60.0+sec/3600.0)/24.0;
    jDate = Gregory(month,day,year)+jTime;

    Precession(jDate,p);
    Ecliptic(jDate,0,earth,v_earth);
    Barycentric(jDate,earth,v_earth);

    if (argc == 3 || argc == 4) {
        printf(
            "\nGeocentric Coordinates       %2d:%02d:%02d      %2d/%02d/%04d\n",
            (int)hour,(int)min,(int)sec,(int)month,(int)day,(int)year);
        printf("\n");
        printf("Object        R.A.       Dec.      Diam.       A.U.\n");
        printf("\n");
    }
    else {
        printf(
            "\nTopocentric Coordinates      %2d:%02d:%02d      %2d/%02d/%04d  |  Longitude %7.2f\n",
            (int)hour,(int)min,(int)sec,(int)month,(int)day,(int)year,
            DM(Deg(LONG)));
        printf("                                                       |  Latitude  %7.2f\n",
            DM(Deg(LAT)));
        printf("Object        R.A.       Dec.      Diam.       A.U.    |  Azimuth  Altitude\n");
        printf("                                                       |\n");
        Topocentric(LONG,LAT,jDate,jTime,earth,v_earth);
    }

    Heliocentric(h_earth,earth);
    Rotate(h_earth,p);

    for (k=0; k<=8; k++) {

        if (k == 0) {
            object[0] = 0.0;
            object[1] = 0.0;
            object[2] = 0.0;
            v_object[0] = 0.0;
            v_object[1] = 0.0;
            v_object[2] = 0.0;
        }
        else
            Ecliptic(jDate,k,object,v_object);

        AU = Distance(earth,object);
        Aberration(object,v_object,v_earth,object,earth);
        Heliocentric(h_object,object);
        Rotate(h_object,p);
        RaDec(h_earth,h_object,&RA,&DEC);
        Nutation(jDate,&RA,&DEC);

        if (argc == 3 || argc == 4) {
            printf("%-12s%8.5f   %8.4f   %8.6f   %8.5f\n",
                    name[k],DMS(Deg(RA)/15),DMS(Deg(DEC)),
                    DMS(DD(sd[k])*2.0/Distance(h_earth,h_object)),AU);
            if (k == 0) {
                Moon(jDate, &RA, &DEC, &AU);
                printf("%-12s%5.2f---   %6.2f--   %6.4f--   %8.5f\n",
                       "Moon",DM(Deg(RA)/15),DM(Deg(DEC)),
                       DMS(2.0*Deg(0.2725*asin(1/AU*ER2AU))),AU);
            }
        }
        else {
            AltAz(RA,DEC,LONG,LAT,jDate,jTime,&AZ,&ALT);
            printf("%-12s%8.5f   %8.4f   %8.6f   %8.5f  |   %6.2f    %6.2f\n",
                    name[k],DMS(Deg(RA)/15),DMS(Deg(DEC)),
                    DMS(DD(sd[k])*2.0/Distance(h_earth,h_object)),AU,
                    DM(Deg(AZ)),DM(Deg(ALT)));
            if (k == 0) {
                Moon(jDate, &RA, &DEC, &AU);
                Moon1(&RA,&DEC,&AU,LONG,LAT,jDate,jTime);
                AltAz(RA,DEC,LONG,LAT,jDate,jTime,&AZ,&ALT);
                printf("%-12s%5.2f---   %6.2f--   %6.4f--   %8.5f  |   %6.2f    %6.2f\n",
                       "Moon",DM(Deg(RA)/15),DM(Deg(DEC)),
                       DMS(2.0*Deg(0.2725*asin(1/AU*ER2AU))),AU,
                       DM(Deg(AZ)),DM(Deg(ALT)));
            }
        }
    }

    printf("\n\n");
}
